<?php
/**
 * Attendance API - Handle check-in and check-out
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../config/google_sheets.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!isset($input['token']) || !isset($input['action'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Token and action are required']);
    exit();
}

$token = sanitize($input['token']);
$action = sanitize($input['action']);

// Validate action
if (!in_array($action, ['check-in', 'check-out'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid action. Must be check-in or check-out']);
    exit();
}

try {
    $pdo = getDB();
    
    // Get employee by token
    $stmt = $pdo->prepare("SELECT id, name, department FROM employees WHERE qr_token = ?");
    $stmt->execute([$token]);
    $employee = $stmt->fetch();
    
    if (!$employee) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Invalid token']);
        exit();
    }
    
    $employeeId = $employee['id'];
    $employeeName = $employee['name'];
    $employeeDepartment = $employee['department'];
    $today = date('Y-m-d');
    $currentTime = date('H:i:s');
    
    // Check existing attendance for today
    $stmt = $pdo->prepare("SELECT id, check_in_time, check_out_time FROM attendance WHERE employee_id = ? AND date = ?");
    $stmt->execute([$employeeId, $today]);
    $attendance = $stmt->fetch();
    
    if ($action === 'check-in') {
        // Check if already checked in today
        if ($attendance && $attendance['check_in_time']) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'You have already checked in today']);
            exit();
        }
        
        // Create or update attendance record
        if ($attendance) {
            // Update existing record
            $stmt = $pdo->prepare("UPDATE attendance SET check_in_time = ? WHERE id = ?");
            $stmt->execute([$currentTime, $attendance['id']]);
        } else {
            // Create new record
            $stmt = $pdo->prepare("INSERT INTO attendance (employee_id, date, check_in_time) VALUES (?, ?, ?)");
            $stmt->execute([$employeeId, $today, $currentTime]);
        }
        
        // Sync to Google Sheets
        appendToGoogleSheets($employeeName, $employeeDepartment, $today, $currentTime, '');
        
        echo json_encode([
            'success' => true,
            'message' => 'Check-in successful',
            'time' => $currentTime,
            'hasCheckedIn' => true
        ]);
        
    } else if ($action === 'check-out') {
        // Check if checked in today
        if (!$attendance || !$attendance['check_in_time']) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'You must check in before checking out']);
            exit();
        }
        
        // Check if already checked out
        if ($attendance['check_out_time']) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'You have already checked out today']);
            exit();
        }
        
        // Update check-out time
        $stmt = $pdo->prepare("UPDATE attendance SET check_out_time = ? WHERE id = ?");
        $stmt->execute([$currentTime, $attendance['id']]);
        
        // Sync to Google Sheets (update existing row)
        appendToGoogleSheets($employeeName, $employeeDepartment, $today, $attendance['check_in_time'], $currentTime);
        
        echo json_encode([
            'success' => true,
            'message' => 'Check-out successful',
            'time' => $currentTime,
            'hasCheckedOut' => true
        ]);
    }
    
} catch (PDOException $e) {
    error_log("Attendance error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Server error. Please try again.']);
}
